"use client";
import React from 'react';
import { useState, useEffect } from 'react';
import Progress from '@/components/Progress';
import { useRouter } from 'next/navigation';
import { AiOutlineReload } from 'react-icons/ai';
import SeoResult from '@/components/SeoResult';
import { BiRightArrowAlt } from 'react-icons/bi';
import Footer from '@/components/Footer';

// Define the Home component
const Home = () => {
  // State variables
  const [url, setUrl] = useState('');
  const [searchUrl, setSearchUrl] = useState('');
  const [seoResult, setSeoResult] = useState(null);
  const [isLoading, setIsLoading] = useState(false);
  const [showAudits, setShowAudits] = useState(false);
  const [isScrolled, setIsScrolled] = useState(false);
  const router = useRouter();

  // useEffect to handle initial setup and clean-up
  useEffect(() => {
    // Handle scroll events to determine if the page is scrolled
    const handleScroll = () => {
      const scrollTop = window.pageYOffset;
      setIsScrolled(scrollTop > 0);
    };

    // Add scroll event listener
    window.addEventListener('scroll', handleScroll);

    // Clean up by removing the scroll event listener
    return () => {
      window.removeEventListener('scroll', handleScroll);
    };
  }, [url]);

  // Function to perform SEO analysis
  const analyzeSeo = async () => {
    // Check if the URL is empty
    if (!url) {
      alert('Please enter a URL');
      return;
    }

    // Validate the URL format
    const urlRegex = /^[^\s.]+(\.[^\s.]+)+$/;
    if (!urlRegex.test(url)) {
      alert('Invalid URL');
      return;
    }

    // Set loading state
    setIsLoading(true);

    try {
      // Define API route for SEO analysis
      const apiRoute = `/api/seo?url=${url}`;
      
      // Fetch data from the API
      const response = await fetch(apiRoute);
      
      // Parse the response JSON and set the SEO result state
      const data = await response.json();
      setSeoResult(data);
    } catch (error) {
      // Handle errors and set loading state to false
      console.error('Error analyzing SEO:', error);
      alert('An error occurred while analyzing SEO. Please try again.');
    } finally {
      setIsLoading(false);
    }
  };

  // Event handler for input change
  const handleInputChange = (e) => {
    setUrl(e.target.value);
  };

  // JSX structure for the Home component
  return (
    <div>
      {/* Display message when not loading and no SEO result */}
      {!isLoading && !seoResult && (
        <div className="text-center mt-[30vh] px-8">
          <h1 className="font-bold text-4xl sm:text-5xl">SEO Checker</h1>
          <p className="mt-4 text-gray-400">
            Get a <b>FREE</b> SEO Report, Get more visitors to your site.
          </p>
        </div>
      )}
      {/* Main content container */}
      <div className="max-w-3xl sm:p-4 mx-auto bg-black min-h-screen">
        {/* Search form */}
        <div
          className={`p-4 sticky top-0 z-40 ${isScrolled &&
            'bg-transparent backdrop-blur-xl border-b border-b-gray-600 sm:border-b-0'}`}
        >
          <div className="mx-auto shadow-blue-light flex items-center bg-transparent rounded-full p-1 border border-blue-600 overflow-hidden w-full max-w-[800px]">
            <label htmlFor="simple-search" className="sr-only">
              Search
            </label>
            <div className="flex flex-grow">
              <div className="text-white flex items-center pl-3 pointer-events-none">
                <svg
                  className="w-5 h-5 text-white"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                  xmlns="http://www.w3.org/2000/svg"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth="2"
                    d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"
                  ></path>
                </svg>
              </div>
              <input
                type="text"
                id="simple-search"
                name="url"
                className="flex-grow bg-transparent backdrop-blur-lg text-white text-sm block w-full p-3 hover:outline-none focus:outline-none"
                placeholder="Enter URL..."
                defaultValue={url}
                onChange={handleInputChange}
                required
              />
            </div>
            <button
              type="submit"
              className="p-3 text-sm font-medium text-white rounded-full bg-blue-700 hover:ring-4 hover:outline-none hover:ring-blue-500/40 bg-opacity-90 backdrop-blur-lg"
              onClick={analyzeSeo}>
              <BiRightArrowAlt className="text-2xl" />
              <span className="sr-only">Search</span>
            </button>
          </div>
        </div>

        {/* Loading state */}
        {isLoading && (
          <div className="text-center p-8">
            <svg
              className="animate-spin h-8 w-8 mx-auto mb-4 text-white mt-1/3"
              viewBox="0 0 24 24"
            >
              <circle
                className="opacity-25"
                cx="12"
                cy="12"
                r="10"
                stroke="currentColor"
                strokeWidth="4"
              />
              <path
                className="opacity-75"
                fill="currentColor"
                d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
              />
            </svg>
            <p className="text-white">Analyzing SEO...</p>
          </div>
        )}

        {/* SEO analysis error state */}
        {seoResult && seoResult.error && (
          <div className="p-12 sm:px-12 sm:py-20">
            <h1 className="text-2xl font-extrabold">An Error Occurred.</h1>
            <p className="mt-2 text-gray-400">
              {seoResult.error} for {searchUrl}.
            </p>
          </div>
        )}

        {/* Display SEO result */}
        {!isLoading && seoResult && !seoResult.error && (
          <div className="mt-8 sm:p-8">
            <center>
              <div className={`grid ${seoResult.domain ? 'sm:grid-cols-2' : ''} p-4 items-center mx-auto`}>
                <div className="mb-8 sm:mb-0">
                  <Progress percent={seoResult.data.score} />
                  <b className="block mt-2">SEO Score</b>
                  <p className="text-gray-400">Values are estimated and may vary</p>
                  <div className="text-center text-gray-400 text-sm font-semibold mt-4 flex items-center justify-center">
                    <span className="flex items-center">
                      <span className="inline-block h-3 w-3 mr-2 rounded-full border border-2 border-red-500"></span>
                      0-49
                    </span>
                    <span className="flex items-center">
                      <span className="inline-block h-3 w-3 mr-2 ml-2 rounded-full border border-4 border-orange-500"></span>
                      50-90
                    </span>
                    <span className="flex items-center">
                      <span className="inline-block h-3 w-3 mr-2 ml-2 rounded-full bg-green-500"></span>
                      90-100
                    </span>
                  </div>
                </div>
                <div className={`${!seoResult.domain ? 'hidden' : ''}`}>
                  <img
                    width={300}
                    height={200}
                    className="rounded-xl p-2 bg-gray-500 rounded-xl"
                    src={`http://image.thum.io/get/width/300/crop/600/${url.includes('://') ? url : 'http://' + url}`} alt="Thumbnail" />
                </div>
              </div>
            </center>

            <hr className='my-12 w-24 mx-auto'/>

            {/* SEO Audits section */}
            <div className='bg-gray-800 sm:rounded-xl p-4 px-6'>
              <h3 className="mb-2 text-md font-bold">
                {/* Toggle button for Audits */}
                <div
                  className="flex w-full font-bold mb-2 focus:outline-none justify-between items-center text-left"
                >
                  <p className='flex items-center'>AUDITS <span className='font-light text-gray-400 ml-2'>({seoResult.data.metrics.total})</span></p>
                </div>
              </h3>
              <div className="block">
                <SeoResult seoResult={seoResult} pageSpeed={true}/>
              </div>
            </div>

            <br />
            <br />

            {/* Report information */}
            <div className='bg-gray-700 sm:rounded-xl p-4 px-6 flex justify-between items-center'>
              <div>
                <h3>Report From {seoResult.data.time}</h3>
              </div>
              <div>
                <AiOutlineReload onClick={router.reload}/>
              </div>
            </div>
          </div>
        )}
      </div>
      {/* Footer component */}
      <Footer />
    </div>
  );
};

// Export the Home component
export default Home;    
